////////////////////////////////////////////////////////////////
//	File			:	"CBitmapFontManager.h"
//
//	Author			:	David Brown (DB)
//
//	Creation Date	:	9/05/2008
//
//	Purpose			:	To handle the loading of fonts and
//						creating many instances of a single loaded
//						font.
/////////////////////////////////////////////////////////////////
#pragma once
#include <windows.h> // for RECT

#include <string>
using std::string;

#include <vector>
using std::vector;

#include "CBitmapFont.h"

//	Holds the read only data common to all instances of the same font.
//	This is the "template."
class CBitmapFontData
{
private:
	string		m_strFileName;

	int			m_nFontImageID;
	int			m_nNumCols;
	int			m_nCharacterWidth;
	int			m_nCharacterHeight;

	char		m_cStartChar;		//	first char on the bitmap font sheet

	std::vector<int> m_kerning;	// Spacing between each character

	inline	void SetFontImageID(int nImageID)	{ m_nFontImageID = nImageID; }
	inline	void SetNumCols(int nNumCols)		{ m_nNumCols = nNumCols; }
	inline	void SetCharWidth(int nWidth)		{ m_nCharacterWidth = nWidth; }
	inline	void SetCharHeight(int nHeight)		{ m_nCharacterHeight = nHeight; }
	inline	void SetStartChar(char ch)			{ m_cStartChar = ch;	}

public:
	CBitmapFontData(void);
	~CBitmapFontData(void);

	bool Load(const char* szFileName);

	inline const char* GetFileName(void) const	{ return m_strFileName.c_str(); }
	inline	int GetFontImageID(void) const		{ return m_nFontImageID;	}
	inline	int	GetNumCols(void) const			{ return m_nNumCols; }
	inline	int GetCharWidth(void) const		{ return m_nCharacterWidth; }
	inline	int GetCharHeight(void) const		{ return m_nCharacterHeight; }
	inline	char GetStartChar(void) const		{ return m_cStartChar; }
	int GetKernWidth(char letter) const { if(!m_kerning.empty()) return m_kerning[letter]; return 0; }

	RECT CalculateRect(int id);
	
	void DrawString(const char* szText, int nPosX, int nPosY, float fScaleX, float fScaleY, DWORD dwColor);

	void LoadKerning(const char* szFilename);
};

//	Singleton class responsible for loading and instancing fonts.
class CBitmapFontManager
{
private:

	vector<CBitmapFontData*>	m_vpData;
	vector<int>					m_vnPrevUsedIDs;

	CBitmapFontManager(void);
	CBitmapFontManager(const CBitmapFontManager&);
	CBitmapFontManager& operator=(const CBitmapFontManager&);

	~CBitmapFontManager(void);

	//	Returns the index of the resource if it is already loaded, -1 otherwise
	int CheckIfAlreadyExists(const char* szFileName);

public:
	static CBitmapFontManager* GetInstance(void);

	//	Loads a font and returns a template id
	int LoadTemplate(const char* szFileName);

	//	Unloads a template
	void UnloadTemplate(int nID);

	//	Given a template id, returns an instance (a copy) of the template
	CBitmapFont CreateTemplateInstance(int nID);

	//	Unloads all stored template fonts
	void UnloadAllTemplates(void);
};